/**
 * VINSpot - VIN Detection and IME Integration
 * by ICB Tech Solutions
 * 
 * Detects VINs on auction websites and provides quick access to IME evaluation
 */

(function() {
    'use strict';

    // Configuration
    const CONFIG = {
        IME_URL: 'https://app.imedealer.com/cms/login.php',
        SCAN_DELAY: 500,            // Delay before scanning (ms)
        RESCAN_INTERVAL: 2000,      // Rescan interval for dynamic content (ms)
        DEBUG: false                 // Set to true for console logging
    };
    
    console.log('🎯 VINSpot: Extension loaded');

    // Track processed VINs to avoid duplicates
    const processedVINs = new Set();
    
    // VIN validation weights for check digit calculation
    const TRANSLITERATION = {
        'A': 1, 'B': 2, 'C': 3, 'D': 4, 'E': 5, 'F': 6, 'G': 7, 'H': 8,
        'J': 1, 'K': 2, 'L': 3, 'M': 4, 'N': 5, 'P': 7, 'R': 9,
        'S': 2, 'T': 3, 'U': 4, 'V': 5, 'W': 6, 'X': 7, 'Y': 8, 'Z': 9,
        '0': 0, '1': 1, '2': 2, '3': 3, '4': 4, '5': 5, '6': 6, '7': 7, '8': 8, '9': 9
    };
    
    const POSITION_WEIGHTS = [8, 7, 6, 5, 4, 3, 2, 10, 0, 9, 8, 7, 6, 5, 4, 3, 2];

    /**
     * Validate VIN structure - first character must be valid country code AND check digit must validate
     */
    function isValidVINStructure(vin) {
        if (!vin || vin.length !== 17) return false;
        
        const upperVIN = vin.toUpperCase();
        
        // Check for invalid characters (I, O, Q)
        if (/[IOQ]/i.test(upperVIN)) return false;
        
        // First character must be a valid World Manufacturer Identifier country code
        // 1-5 = North America, J = Japan, K = Korea, L = China, S = UK, T = Switzerland,
        // V = France/Spain, W = Germany, Y = Sweden/Finland, Z = Italy, 2 = Canada, 3 = Mexico
        const validFirstChars = ['1', '2', '3', '4', '5', 'J', 'K', 'L', 'S', 'T', 'V', 'W', 'X', 'Y', 'Z'];
        if (!validFirstChars.includes(upperVIN[0])) {
            console.log('🎯 VINSpot: Rejected - invalid first char:', upperVIN[0], 'in', upperVIN);
            return false;
        }
        
        // Validate check digit (position 9, index 8)
        const transliteration = {
            'A':1,'B':2,'C':3,'D':4,'E':5,'F':6,'G':7,'H':8,
            'J':1,'K':2,'L':3,'M':4,'N':5,'P':7,'R':9,
            'S':2,'T':3,'U':4,'V':5,'W':6,'X':7,'Y':8,'Z':9,
            '0':0,'1':1,'2':2,'3':3,'4':4,'5':5,'6':6,'7':7,'8':8,'9':9
        };
        const weights = [8,7,6,5,4,3,2,10,0,9,8,7,6,5,4,3,2];
        
        let sum = 0;
        for (let i = 0; i < 17; i++) {
            const char = upperVIN[i];
            const value = transliteration[char];
            if (value === undefined) {
                console.log('🎯 VINSpot: Rejected - invalid char:', char, 'in', upperVIN);
                return false;
            }
            sum += value * weights[i];
        }
        
        const remainder = sum % 11;
        const expectedCheckDigit = remainder === 10 ? 'X' : remainder.toString();
        const actualCheckDigit = upperVIN[8];
        
        if (actualCheckDigit !== expectedCheckDigit) {
            console.log('🎯 VINSpot: Rejected - check digit mismatch. Expected:', expectedCheckDigit, 'Got:', actualCheckDigit, 'in', upperVIN);
            return false;
        }
        
        console.log('🎯 VINSpot: Valid VIN confirmed:', upperVIN);
        return true;
    }

    /**
     * Find VINs in text content
     */
    function findVINsInText(text) {
        // VIN pattern: 17 alphanumeric characters excluding I, O, Q
        const vinPattern = /\b[A-HJ-NPR-Z0-9]{17}\b/gi;
        const matches = text.match(vinPattern) || [];
        
        if (CONFIG.DEBUG && matches.length > 0) {
            console.log('🎯 VINSpot: Potential VINs found:', matches);
        }
        
        // Filter to valid VINs - be lenient, accept most 17-char matches
        return matches.filter(vin => {
            const upperVIN = vin.toUpperCase();
            // Basic validation: check it doesn't contain I, O, Q
            if (/[IOQ]/i.test(upperVIN)) {
                if (CONFIG.DEBUG) console.log('🎯 VINSpot: Rejected (IOQ):', upperVIN);
                return false;
            }
            // Accept any 17-char alphanumeric that passes basic check
            return true;
        });
    }

    /**
     * Extract mileage from text using multiple patterns
     * @param {string} text - Text to search
     * @param {number} maxLength - Maximum text length to process (default 500)
     */
    function extractMileage(text, maxLength = 500) {
        if (!text) return null;
        
        // Only process reasonable length text to avoid picking up random numbers
        if (text.length > maxLength) {
            console.log('🎯 VINSpot: Skipping mileage extraction - text too long:', text.length, 'chars (max:', maxLength + ')');
            return null;
        }
        
        // Pattern 1: "73,925mi" or "73925mi" or "73,925 mi" (most common on Manheim)
        const miPattern = /(\d{1,3}(?:,\d{3})*|\d+)\s*mi(?:les)?(?!\w)/i;
        let match = text.match(miPattern);
        if (match) {
            const mileage = parseInt(match[1].replace(/,/g, ''), 10);
            console.log('🎯 VINSpot: miPattern matched:', match[0], '-> mileage:', mileage);
            // Validate reasonable mileage (0 to 999,999)
            if (mileage >= 0 && mileage <= 999999) {
                return mileage;
            }
        }
        
        // Pattern 2: "Odo: 12345" or "ODO 12345" or "Odo:12345"
        const odoPattern = /(?:odo|odometer)[:\s]*(\d{1,3}(?:,\d{3})*|\d+)/i;
        match = text.match(odoPattern);
        if (match) {
            const mileage = parseInt(match[1].replace(/,/g, ''), 10);
            console.log('🎯 VINSpot: odoPattern matched:', match[0], '-> mileage:', mileage);
            if (mileage >= 0 && mileage <= 999999) {
                return mileage;
            }
        }
        
        // Pattern 3: "Mileage: 12345" or "Miles: 12345"
        const mileagePattern = /(?:mileage|miles)[:\s]*(\d{1,3}(?:,\d{3})*|\d+)/i;
        match = text.match(mileagePattern);
        if (match) {
            const mileage = parseInt(match[1].replace(/,/g, ''), 10);
            console.log('🎯 VINSpot: mileagePattern matched:', match[0], '-> mileage:', mileage);
            if (mileage >= 0 && mileage <= 999999) {
                return mileage;
            }
        }
        
        return null;
    }
    
    /**
     * Search page for mileage using DOM queries (for detail pages)
     * This is a broader search when local element traversal fails
     */
    function findMileageOnPage() {
        console.log('🎯 VINSpot: Searching page for mileage...');
        
        // Pattern to match mileage in text
        const mileageRegex = /(\d{1,3}(?:,\d{3})*|\d+)\s*mi(?:les)?(?!\w)/i;
        
        // Get all text-containing elements (but not scripts, styles, etc.)
        const walker = document.createTreeWalker(
            document.body,
            NodeFilter.SHOW_TEXT,
            {
                acceptNode: function(node) {
                    const parent = node.parentElement;
                    if (!parent) return NodeFilter.FILTER_REJECT;
                    const tag = parent.tagName.toLowerCase();
                    if (['script', 'style', 'noscript', 'textarea', 'input'].includes(tag)) {
                        return NodeFilter.FILTER_REJECT;
                    }
                    // Only accept nodes with potential mileage text
                    if (mileageRegex.test(node.textContent)) {
                        return NodeFilter.FILTER_ACCEPT;
                    }
                    return NodeFilter.FILTER_REJECT;
                }
            }
        );
        
        const mileageCandidates = [];
        let node;
        while (node = walker.nextNode()) {
            const text = node.textContent.trim();
            const match = text.match(mileageRegex);
            if (match) {
                const mileage = parseInt(match[1].replace(/,/g, ''), 10);
                if (mileage >= 100 && mileage <= 999999) { // Reasonable mileage range
                    mileageCandidates.push({
                        mileage: mileage,
                        text: text,
                        element: node.parentElement
                    });
                    console.log('🎯 VINSpot: Found mileage candidate:', mileage, 'in text:', text.substring(0, 100));
                }
            }
        }
        
        if (mileageCandidates.length === 1) {
            console.log('🎯 VINSpot: Single mileage found on page:', mileageCandidates[0].mileage);
            return mileageCandidates[0].mileage;
        } else if (mileageCandidates.length > 1) {
            // Multiple candidates - try to pick the most likely one
            // Prefer candidates in elements with relevant class names
            for (const candidate of mileageCandidates) {
                const el = candidate.element;
                const classAndId = (el.className + ' ' + el.id + ' ' + (el.parentElement?.className || '')).toLowerCase();
                if (classAndId.includes('spec') || classAndId.includes('detail') || 
                    classAndId.includes('info') || classAndId.includes('odometer') ||
                    classAndId.includes('mileage') || classAndId.includes('vehicle')) {
                    console.log('🎯 VINSpot: Selected mileage from relevant container:', candidate.mileage);
                    return candidate.mileage;
                }
            }
            // Default to first candidate if no better match
            console.log('🎯 VINSpot: Multiple mileages found, using first:', mileageCandidates[0].mileage);
            return mileageCandidates[0].mileage;
        }
        
        console.log('🎯 VINSpot: No mileage found on page');
        return null;
    }

    /**
     * Search for mileage near a VIN element
     */
    function findMileageNearElement(element) {
        // Strategy 1: Look for mileage in the same container/row first (most reliable)
        // Check siblings and nearby elements for "XXmi" pattern
        const parent = element.parentElement;
        if (parent) {
            // Check parent's direct text
            const parentText = parent.innerText || parent.textContent || '';
            // Only check if parent text is reasonably short (not entire page)
            if (parentText.length < 300) {
                const mileage = extractMileage(parentText);
                if (mileage) {
                    console.log('🎯 VINSpot: Found mileage', mileage, 'in parent');
                    return mileage;
                }
            }
        }
        
        // Strategy 2: Look in grandparent (common for table rows or cards)
        const grandparent = element.parentElement?.parentElement;
        if (grandparent) {
            const gpText = grandparent.innerText || grandparent.textContent || '';
            if (gpText.length < 500) {
                const mileage = extractMileage(gpText);
                if (mileage) {
                    console.log('🎯 VINSpot: Found mileage', mileage, 'in grandparent');
                    return mileage;
                }
            }
        }
        
        // Strategy 3: Look for table row
        const row = element.closest('tr');
        if (row) {
            const mileage = extractMileage(row.innerText || row.textContent || '');
            if (mileage) {
                console.log('🎯 VINSpot: Found mileage', mileage, 'in table row');
                return mileage;
            }
        }
        
        // Strategy 4: Look for common card/listing container classes
        const card = element.closest('[class*="card"], [class*="listing"], [class*="vehicle"], [class*="item"], [class*="result"]');
        if (card) {
            const cardText = card.innerText || card.textContent || '';
            if (cardText.length < 1000) {
                const mileage = extractMileage(cardText);
                if (mileage) {
                    console.log('🎯 VINSpot: Found mileage', mileage, 'in card container');
                    return mileage;
                }
            }
        }
        
        // Strategy 5: Search the element's own text
        const ownText = element.innerText || element.textContent || '';
        if (ownText.length < 200) {
            const mileage = extractMileage(ownText);
            if (mileage) {
                console.log('🎯 VINSpot: Found mileage', mileage, 'in element text');
                return mileage;
            }
        }
        
        // Strategy 6: ADESA/Openlane detail page - look for preceding sibling elements
        // On detail pages, VIN is often below the specs line that contains mileage
        let searchElement = element.parentElement;
        for (let i = 0; i < 5 && searchElement; i++) {
            // Check all previous siblings of the current element
            let sibling = searchElement.previousElementSibling;
            while (sibling) {
                const sibText = sibling.innerText || sibling.textContent || '';
                if (sibText.length < 300) {
                    const mileage = extractMileage(sibText);
                    if (mileage) {
                        console.log('🎯 VINSpot: Found mileage', mileage, 'in previous sibling');
                        return mileage;
                    }
                }
                sibling = sibling.previousElementSibling;
            }
            searchElement = searchElement.parentElement;
        }
        
        // Strategy 7: Look for ADESA/auction-specific containers (with higher text limit)
        const adesaSelectors = [
            '[class*="detail"]',
            '[class*="summary"]',
            '[class*="specs"]',
            '[class*="info"]',
            '[class*="description"]',
            '[class*="header"]',
            'article',
            'section',
            'main'
        ];
        
        for (const selector of adesaSelectors) {
            const container = element.closest(selector);
            if (container) {
                const containerText = container.innerText || container.textContent || '';
                if (containerText.length < 3000) {
                    const mileage = extractMileage(containerText, 3000); // Allow larger text
                    if (mileage) {
                        console.log('🎯 VINSpot: Found mileage', mileage, 'in', selector, 'container');
                        return mileage;
                    }
                }
            }
        }
        
        // Strategy 8: Walk up further and search broader area (detail page fallback)
        let ancestor = element.parentElement;
        for (let level = 0; level < 10 && ancestor; level++) {
            const ancestorText = ancestor.innerText || ancestor.textContent || '';
            // Allow larger text for higher-level ancestors on detail pages
            const maxLength = 1000 + (level * 500);
            if (ancestorText.length < maxLength && ancestorText.length > 0) {
                const mileage = extractMileage(ancestorText, maxLength); // Pass the limit
                if (mileage) {
                    console.log('🎯 VINSpot: Found mileage', mileage, 'at ancestor level', level);
                    return mileage;
                }
            }
            ancestor = ancestor.parentElement;
        }
        
        // Strategy 9: Page-wide search (final fallback for detail pages)
        // This searches the entire page DOM for mileage patterns
        const pageMileage = findMileageOnPage();
        if (pageMileage) {
            return pageMileage;
        }
        
        console.log('🎯 VINSpot: No mileage found near VIN');
        return null;
    }

    /**
     * Create VINSpot icon element
     */
    function createVINSpotIcon(vin, mileage) {
        const icon = document.createElement('span');
        icon.className = 'vinspot-icon';
        icon.setAttribute('data-vin', vin);
        icon.setAttribute('data-mileage', mileage || '');
        icon.setAttribute('title', `VINSpot: Click to evaluate\nVIN: ${vin}\nMileage: ${mileage ? mileage.toLocaleString() : 'Not detected'}`);
        
        // Add inline styles as fallback
        icon.style.cssText = `
            display: inline-flex !important;
            align-items: center !important;
            justify-content: center !important;
            width: 16px !important;
            height: 16px !important;
            min-width: 16px !important;
            min-height: 16px !important;
            margin-left: 4px !important;
            cursor: pointer !important;
            vertical-align: middle !important;
            border-radius: 50% !important;
            background: linear-gradient(135deg, #39ff14 0%, #00c853 100%) !important;
            box-shadow: 0 2px 6px rgba(57, 255, 20, 0.5) !important;
            position: relative !important;
            z-index: 999999 !important;
        `;
        
        // SVG icon - crosshair/target design
        icon.innerHTML = `
            <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="width:12px;height:12px;color:#0a0a0a;">
                <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2"/>
                <circle cx="12" cy="12" r="6" stroke="currentColor" stroke-width="1.5"/>
                <circle cx="12" cy="12" r="2" fill="currentColor"/>
                <line x1="12" y1="0" x2="12" y2="4" stroke="currentColor" stroke-width="2"/>
                <line x1="12" y1="20" x2="12" y2="24" stroke="currentColor" stroke-width="2"/>
                <line x1="0" y1="12" x2="4" y2="12" stroke="currentColor" stroke-width="2"/>
                <line x1="20" y1="12" x2="24" y2="12" stroke="currentColor" stroke-width="2"/>
            </svg>
        `;
        
        // Click handler
        icon.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const vinValue = this.getAttribute('data-vin');
            const mileageValue = this.getAttribute('data-mileage');
            
            // Build CMS login URL with VIN/mileage parameters
            // These params will persist through the login process and pre-populate the IME tool
            let url = `${CONFIG.IME_URL}?vin=${encodeURIComponent(vinValue)}`;
            if (mileageValue) {
                url += `&mileage=${encodeURIComponent(mileageValue)}`;
            }
            url += '&source=vinspot';
            
            // Open in new window
            window.open(url, '_blank', 'width=1400,height=900,scrollbars=yes,resizable=yes');
            
            // Visual feedback
            this.style.transform = 'scale(0.9)';
            setTimeout(() => this.style.transform = '', 200);
            
            console.log('🎯 VINSpot: Opening IME for', vinValue, 'with mileage', mileageValue);
        });
        
        return icon;
    }

    /**
     * Find text nodes containing VINs
     */
    function findVINTextNodes(root = document.body) {
        const vinNodes = [];
        const walker = document.createTreeWalker(
            root,
            NodeFilter.SHOW_TEXT,
            {
                acceptNode: function(node) {
                    // Skip script, style, and already processed elements
                    const parent = node.parentElement;
                    if (!parent) return NodeFilter.FILTER_REJECT;
                    
                    const tagName = parent.tagName.toLowerCase();
                    if (['script', 'style', 'noscript', 'textarea', 'input'].includes(tagName)) {
                        return NodeFilter.FILTER_REJECT;
                    }
                    
                    // Skip VINSpot elements
                    if (parent.closest('.vinspot-icon, .vinspot-wrapper')) {
                        return NodeFilter.FILTER_REJECT;
                    }
                    
                    // Check if contains potential VIN
                    const text = node.textContent;
                    if (text && text.length >= 17 && /[A-HJ-NPR-Z0-9]{17}/i.test(text)) {
                        return NodeFilter.FILTER_ACCEPT;
                    }
                    
                    return NodeFilter.FILTER_REJECT;
                }
            }
        );
        
        let node;
        while (node = walker.nextNode()) {
            vinNodes.push(node);
        }
        
        if (CONFIG.DEBUG) {
            console.log('🎯 VINSpot: Found', vinNodes.length, 'text nodes with potential VINs');
        }
        
        return vinNodes;
    }
    
    /**
     * Alternative: Scan all elements for VINs (more aggressive)
     */
    function scanAllElements() {
        const vinPattern = /[A-HJ-NPR-Z0-9]{17}/g;
        
        // Find elements that might contain VINs - target small containers
        const potentialElements = document.querySelectorAll('div, span, td, p, a, li, b, strong');
        
        console.log('🎯 VINSpot: Scanning', potentialElements.length, 'elements');
        
        let foundCount = 0;
        
        potentialElements.forEach(el => {
            // Skip if already processed or is our element
            if (el.classList.contains('vinspot-processed') || 
                el.classList.contains('vinspot-wrapper') ||
                el.classList.contains('vinspot-icon')) return;
            
            // Skip if any ancestor is already processed
            if (el.closest('.vinspot-processed')) return;
            
            // Skip if already has an icon nearby
            if (el.querySelector('.vinspot-icon')) return;
            
            // Get text content
            const fullText = el.textContent || '';
            
            // Skip if too long (container) or too short
            if (fullText.length < 17 || fullText.length > 500) return;
            
            // Look for VIN
            vinPattern.lastIndex = 0;
            const match = vinPattern.exec(fullText);
            
            if (!match) return;
            
            const vin = match[0].toUpperCase();
            
            // Skip if contains invalid chars
            if (/[IOQ]/i.test(vin)) return;
            
            // Validate VIN structure (first char = country, check digit)
            if (!isValidVINStructure(vin)) return;
            
            // Check if THIS ELEMENT already has an icon (not page-wide)
            if (el.querySelector('.vinspot-icon[data-vin="' + vin + '"]')) return;
            
            console.log('🎯 VINSpot: Found VIN:', vin);
            
            // Find mileage
            const mileage = findMileageNearElement(el);
            
            try {
                const icon = createVINSpotIcon(vin, mileage);
                
                // Try to find the exact text node containing the VIN and insert icon right after it
                let inserted = false;
                const walker = document.createTreeWalker(el, NodeFilter.SHOW_TEXT, null, false);
                let node;
                let accumulated = '';
                
                while (node = walker.nextNode()) {
                    const prevLen = accumulated.length;
                    accumulated += node.textContent;
                    
                    // Check if VIN ends in this text node
                    const vinIdx = accumulated.toUpperCase().indexOf(vin);
                    if (vinIdx !== -1) {
                        const vinEndInAccum = vinIdx + 17;
                        
                        // Check if VIN end is within this text node
                        if (vinEndInAccum > prevLen && vinEndInAccum <= accumulated.length) {
                            const offsetInNode = vinEndInAccum - prevLen;
                            const nodeText = node.textContent;
                            
                            // Split the text node and insert icon inline
                            const before = nodeText.substring(0, offsetInNode);
                            const after = nodeText.substring(offsetInNode);
                            
                            const wrapper = document.createElement('span');
                            wrapper.className = 'vinspot-wrapper';
                            wrapper.style.cssText = 'display:inline !important;';
                            wrapper.appendChild(document.createTextNode(before));
                            wrapper.appendChild(icon);
                            if (after) {
                                wrapper.appendChild(document.createTextNode(after));
                            }
                            
                            node.parentNode.replaceChild(wrapper, node);
                            inserted = true;
                            console.log('🎯 VINSpot: ✅ Inserted icon inline after VIN');
                            break;
                        }
                    }
                }
                
                // Fallback: insert after the element
                if (!inserted) {
                    if (el.nextSibling) {
                        el.parentNode.insertBefore(icon, el.nextSibling);
                    } else {
                        el.parentNode.appendChild(icon);
                    }
                    console.log('🎯 VINSpot: ✅ Inserted icon after element (fallback)');
                }
                
                el.classList.add('vinspot-processed');
                foundCount++;
                
            } catch (e) {
                console.error('🎯 VINSpot: ❌ Failed:', e.message);
            }
        });
        
        console.log('🎯 VINSpot: Added', foundCount, 'new icons');
        return foundCount;
    }

    /**
     * Process a text node and inject VINSpot icons
     */
    function processTextNode(textNode) {
        const text = textNode.textContent;
        const vins = findVINsInText(text);
        
        if (vins.length === 0) return;
        
        // Filter out already processed VINs
        const newVINs = vins.filter(vin => !processedVINs.has(vin.toUpperCase()));
        if (newVINs.length === 0) return;
        
        const parent = textNode.parentElement;
        if (!parent) return;
        
        // For each VIN found, inject icon
        newVINs.forEach(vin => {
            const upperVIN = vin.toUpperCase();
            processedVINs.add(upperVIN);
            
            // Find mileage near this element
            const mileage = findMileageNearElement(parent);
            
            // Create wrapper to hold original text + icon
            const vinIndex = text.toUpperCase().indexOf(upperVIN);
            if (vinIndex === -1) return;
            
            try {
                // Split text around VIN
                const beforeText = text.substring(0, vinIndex + 17);
                const afterText = text.substring(vinIndex + 17);
                
                // Create wrapper span
                const wrapper = document.createElement('span');
                wrapper.className = 'vinspot-wrapper';
                
                // Add text before and including VIN
                wrapper.appendChild(document.createTextNode(beforeText));
                
                // Add VINSpot icon
                const icon = createVINSpotIcon(upperVIN, mileage);
                wrapper.appendChild(icon);
                
                // Add remaining text
                if (afterText) {
                    wrapper.appendChild(document.createTextNode(afterText));
                }
                
                // Replace original text node
                parent.replaceChild(wrapper, textNode);
                
                if (CONFIG.DEBUG) {
                    console.log('VINSpot: Found VIN', upperVIN, 'with mileage', mileage);
                }
            } catch (e) {
                if (CONFIG.DEBUG) {
                    console.error('VINSpot: Error processing VIN', upperVIN, e);
                }
            }
        });
    }

    /**
     * Scan page for VINs
     */
    function scanPage() {
        console.log('🎯 VINSpot: === Starting scan ===');
        
        // Count existing icons
        const existingIcons = document.querySelectorAll('.vinspot-icon').length;
        
        // Run the scanner
        const newIcons = scanAllElements();
        
        const totalIcons = document.querySelectorAll('.vinspot-icon').length;
        console.log(`🎯 VINSpot: === Scan done. Before: ${existingIcons}, New: ${newIcons}, Total: ${totalIcons} ===`);
        
        return newIcons;
    }

    /**
     * Set up mutation observer for dynamic content
     */
    function setupObserver() {
        const observer = new MutationObserver((mutations) => {
            let shouldScan = false;
            
            for (const mutation of mutations) {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    for (const node of mutation.addedNodes) {
                        if (node.nodeType === Node.ELEMENT_NODE && 
                            !node.closest('.vinspot-icon, .vinspot-wrapper')) {
                            shouldScan = true;
                            break;
                        }
                    }
                }
                if (shouldScan) break;
            }
            
            if (shouldScan) {
                // Debounce scanning
                clearTimeout(window.vinspotScanTimeout);
                window.vinspotScanTimeout = setTimeout(() => {
                    if (CONFIG.DEBUG) console.log('🎯 VINSpot: DOM changed, rescanning...');
                    scanPage();
                }, 500);
            }
        });
        
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
        
        console.log('🎯 VINSpot: Mutation observer active');
    }

    /**
     * Initialize VINSpot
     */
    function init() {
        // Don't run on IME/CMS itself
        if (window.location.hostname.includes('marketeval.icbtechsolutions.com') ||
            window.location.hostname.includes('app.imedealer.com')) {
            console.log('🎯 VINSpot: Skipping ICB Tech site');
            return;
        }
        
        console.log('🎯 VINSpot: Initializing on', window.location.hostname);
        
        // Initial scan after delay (wait for page to fully render)
        setTimeout(() => {
            console.log('🎯 VINSpot: Running initial scan...');
            scanPage();
            setupObserver();
        }, CONFIG.SCAN_DELAY);
        
        // Periodic rescan for dynamically loaded content (simulcast, infinite scroll, etc.)
        setInterval(() => {
            scanPage();
        }, CONFIG.RESCAN_INTERVAL);
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();
